/**
 * Copyright 2021 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * @fileoverview This module serves routes that begin with "/google". These
 * routes handle Google OAuth.
 */

const express = require('express');
const router = express.Router();
const { getOAuthClient, getAuthUrl } = require('../clients/oauth2client');

// Load the client_secret.json file into a keys object.
const keys = require('../../client_secret.json');

/**
 * Route: /google
 *
 * This route constructs the Google authorization URL then redirects the user
 * to that URL. The authorization URL will prompt the user to sign-in and then
 * if applicable accept scopes.
 */
router.get('/', function (req, res, next) {
  const url = getAuthUrl();
  res.redirect(url);
});

/**
 * Route: /google/callback
 *
 * This route is called by Google at the completion of sign-in and
 * authorization. The request will include an authorization code as a query
 * parameter. This code is then exchanged for offline access credentials.
 */
router.get('/callback', function (req, res, next) {
  const code = req.query.code;

  if (!code) {
    res.redirect('/failed');
  } else {
    async function auth() {
      const oauth2Client = getOAuthClient();
      const { tokens } = await oauth2Client.getToken(code);

      // Verify the tokens we received.
      const ticket = await oauth2Client.verifyIdToken({
        idToken: tokens.id_token,
        audience: keys.web.client_id,
      });

      // After verifying the token we can access the payload.
      const info = ticket.getPayload();

      // Build a user object from payload and tokens.
      let user = {
        sub: info.sub,
        email: info.email,
        displayName: info.name,
        portraitUrl: info.picture,
      };

      // Only add the refresh token if we receive one.
      // We will only receive one the first time a user authenticates the addon
      // so we do not want to clear a previous value. If the refreshToken is
      // lost, the user will need to re-authenticate.
      if (tokens.refresh_token) {
        user.refreshToken = tokens.refresh_token;
      }

      // WARNING : NOT FOR PRODUCTION
      // ----------------------------
      // This is a sample application for development purposes. You should
      // follow best practices when securing your production application
      // and in particular how you securely store and use OAuth tokens.
      //
      // Review these resources for additional security considerations
      // + Google Identity developer website
      //   https://developers.google.com/identity
      // + OAuth 2.0 Security Best Current Practice
      //   https://datatracker.ietf.org/doc/html/draft-ietf-oauth-security-topics
      // + OAuth 2.0 Threat Model and Security Considerations
      //   https://datatracker.ietf.org/doc/html/rfc6819
      //
      // Storing tokens in the session is for demonstration purposes. In your
      // production application be sure to store your tokens securely. If you
      // lose the refresh token, you will need to ask the user to authorize
      // your add-on again to receive a new one.
      req.session.user = user;
      req.session.tokens = tokens;
      console.log('user:', user);

      // Redirect to a page that will close the pop-up window.
      res.redirect('/closepopup');
    }
    auth().catch((err) => next(err));
  }
});

module.exports = router;
